<?php

namespace App\Http\Controllers;

use App\Models\Currency;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class CurrencyController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $currencies = Currency::ordered()->paginate(15);
        $primaryCurrency = Currency::getPrimary();

        return view('adminlte.currencies.index', compact('currencies', 'primaryCurrency'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('adminlte.currencies.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => 'required|string|max:3|unique:currencies,code',
            'symbol' => 'required|string|max:10',
            'is_primary' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        $currency = Currency::create($request->all());

        // If this is set as primary, update other currencies
        if ($request->is_primary) {
            $currency->setAsPrimary();
        }

        return redirect()->route('currencies.index')
                        ->with('success', 'Currency created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Currency $currency)
    {
        return view('adminlte.currencies.show', compact('currency'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Currency $currency)
    {
        return view('adminlte.currencies.edit', compact('currency'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Currency $currency)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'code' => ['required', 'string', 'max:3', Rule::unique('currencies')->ignore($currency->id)],
            'symbol' => 'required|string|max:10',
            'is_primary' => 'boolean',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0',
        ]);

        $currency->update($request->all());

        // If this is set as primary, update other currencies
        if ($request->is_primary) {
            $currency->setAsPrimary();
        }

        return redirect()->route('currencies.index')
                        ->with('success', 'Currency updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Currency $currency)
    {
        // Prevent deletion of primary currency
        if ($currency->is_primary) {
            return redirect()->route('currencies.index')
                            ->with('error', 'Cannot delete the primary currency. Please set another currency as primary first.');
        }

        $currency->delete();

        return redirect()->route('currencies.index')
                        ->with('success', 'Currency deleted successfully!');
    }

    /**
     * Set a currency as primary
     */
    public function setPrimary(Currency $currency)
    {
        $currency->setAsPrimary();

        return redirect()->route('currencies.index')
                        ->with('success', 'Primary currency updated successfully!');
    }

    /**
     * Toggle active status of currency
     */
    public function toggleStatus(Currency $currency)
    {
        // Prevent deactivating primary currency
        if ($currency->is_primary && $currency->is_active) {
            return redirect()->route('currencies.index')
                            ->with('error', 'Cannot deactivate the primary currency. Please set another currency as primary first.');
        }

        $currency->update(['is_active' => !$currency->is_active]);

        $status = $currency->is_active ? 'activated' : 'deactivated';

        return redirect()->route('currencies.index')
                        ->with('success', "Currency {$status} successfully!");
    }
}
