<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Transaction;
use App\Models\Membership;
use App\Models\Loan;
use App\Models\Saving;
use App\Mail\MemberWelcomeEmail;
use App\Mail\RegistrationFeePaymentReceipt;
use Illuminate\Support\Facades\Mail;

class MpesaController extends Controller
{
    public function registrationFeeCallback(Request $request)
    {
        $data = $request->all();

        // Log the callback data
        \Log::info('M-Pesa Registration Fee Callback', $data);

        // Check if the callback contains stkCallback
        if (isset($data['Body']['stkCallback'])) {
            $callbackData = $data['Body']['stkCallback'];

            $merchantRequestId = $callbackData['MerchantRequestID'];
            $resultCode = $callbackData['ResultCode'];
            $resultDesc = $callbackData['ResultDesc'];

            // Find the transaction by MerchantRequestID
            $transaction = Transaction::where('mpesa_transaction_id', $merchantRequestId)->first();

            if ($transaction) {
                if ($resultCode == 0) {
                    // Payment successful - extract M-Pesa receipt number
                    $mpesaReceiptNumber = null;
                    if (isset($callbackData['CallbackMetadata']['Item'])) {
                        foreach ($callbackData['CallbackMetadata']['Item'] as $item) {
                            if ($item['Name'] === 'MpesaReceiptNumber') {
                                $mpesaReceiptNumber = $item['Value'];
                                break;
                            }
                        }
                    }

                    // Update transaction with receipt number and processed timestamp
                    $transaction->update([
                        'status' => 'completed',
                        'mpesa_response' => json_encode($callbackData),
                        'reference_number' => $mpesaReceiptNumber,
                        'processed_at' => now()
                    ]);

                    // Update membership registration fee status
                    $membership = Membership::where('user_id', $transaction->user_id)->first();
                    if ($membership) {
                        $membership->update(['registration_fee_status' => 'paid']);

                        // Send welcome email and payment receipt
                        try {
                            // Send welcome email
                            Mail::to($membership->user->email)->send(new MemberWelcomeEmail($membership->user, $membership));

                            // Send payment receipt
                            Mail::to($membership->user->email)->send(new RegistrationFeePaymentReceipt($membership->user, $membership, $transaction));

                            \Log::info('Welcome email and payment receipt sent successfully', [
                                'user_email' => $membership->user->email,
                                'membership_id' => $membership->id
                            ]);
                        } catch (\Exception $e) {
                            \Log::error('Failed to send emails after registration fee payment', [
                                'user_email' => $membership->user->email,
                                'membership_id' => $membership->id,
                                'error' => $e->getMessage()
                            ]);
                        }
                    }

                    \Log::info('Registration fee payment completed', [
                        'transaction_id' => $transaction->id,
                        'membership_id' => $membership->id ?? null,
                        'mpesa_receipt' => $mpesaReceiptNumber
                    ]);
                } else {
                    // Payment failed
                    $transaction->update([
                        'status' => 'failed',
                        'mpesa_response' => json_encode($callbackData)
                    ]);

                    \Log::warning('Registration fee payment failed', [
                        'transaction_id' => $transaction->id,
                        'result_code' => $resultCode,
                        'result_desc' => $resultDesc
                    ]);
                }
            }
        }

        // Return success response to M-Pesa
        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Callback received successfully']);
    }

    public function loanPaymentCallback(Request $request)
    {
        $data = $request->all();

        // Log the callback data
        \Log::info('M-Pesa Loan Payment Callback', $data);

        // Check if the callback contains stkCallback
        if (isset($data['Body']['stkCallback'])) {
            $callbackData = $data['Body']['stkCallback'];

            $merchantRequestId = $callbackData['MerchantRequestID'];
            $resultCode = $callbackData['ResultCode'];
            $resultDesc = $callbackData['ResultDesc'];

            // Find the transaction by MerchantRequestID
            $transaction = Transaction::where('mpesa_transaction_id', $merchantRequestId)->first();

            if ($transaction && $transaction->type === 'loan_repayment') {
                if ($resultCode == 0) {
                    // Payment successful - extract M-Pesa receipt number
                    $mpesaReceiptNumber = null;
                    if (isset($callbackData['CallbackMetadata']['Item'])) {
                        foreach ($callbackData['CallbackMetadata']['Item'] as $item) {
                            if ($item['Name'] === 'MpesaReceiptNumber') {
                                $mpesaReceiptNumber = $item['Value'];
                                break;
                            }
                        }
                    }

                    // Update transaction with M-Pesa receipt number
                    $transaction->update([
                        'status' => 'completed',
                        'mpesa_response' => json_encode($callbackData),
                        'reference_number' => $mpesaReceiptNumber,
                        'processed_at' => now()
                    ]);

                    // Update loan paid amount
                    $loan = $transaction->loan;
                    if ($loan) {
                        $loan->increment('paid_amount', $transaction->amount);

                        // Check if loan is fully paid
                        if ($loan->balance <= 0) {
                            $loan->update(['status' => 'paid']);
                        }

                        \Log::info('Loan repayment completed via M-Pesa', [
                            'transaction_id' => $transaction->id,
                            'loan_id' => $loan->id,
                            'amount' => $transaction->amount,
                            'mpesa_receipt' => $mpesaReceiptNumber
                        ]);
                    }
                } else {
                    // Payment failed
                    $transaction->update([
                        'status' => 'failed',
                        'mpesa_response' => json_encode($callbackData)
                    ]);

                    \Log::warning('Loan repayment failed via M-Pesa', [
                        'transaction_id' => $transaction->id,
                        'result_code' => $resultCode,
                        'result_desc' => $resultDesc
                    ]);
                }
            }
        }

        // Return success response to M-Pesa
        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Callback received successfully']);
    }

    public function savingsDepositCallback(Request $request)
    {
        $data = $request->all();

        // Log the callback data
        \Log::info('M-Pesa Savings Deposit Callback', $data);

        // Check if the callback contains stkCallback
        if (isset($data['Body']['stkCallback'])) {
            $callbackData = $data['Body']['stkCallback'];

            $merchantRequestId = $callbackData['MerchantRequestID'];
            $resultCode = $callbackData['ResultCode'];
            $resultDesc = $callbackData['ResultDesc'];

            // Find the transaction by MerchantRequestID
            $transaction = Transaction::where('mpesa_transaction_id', $merchantRequestId)->first();

            if ($transaction && $transaction->type === 'savings_deposit') {
                if ($resultCode == 0) {
                    // Payment successful - update savings balance
                    $transaction->update([
                        'status' => 'completed',
                        'mpesa_response' => json_encode($callbackData)
                    ]);

                    // Update savings account balance
                    $saving = Saving::find($transaction->reference_id);
                    if ($saving) {
                        $saving->update([
                            'amount' => $saving->amount + $transaction->amount,
                            'status' => $saving->status === 'pending' ? 'active' : $saving->status
                        ]);
                    }

                    \Log::info('Savings deposit completed', [
                        'transaction_id' => $transaction->id,
                        'savings_id' => $transaction->reference_id,
                        'amount' => $transaction->amount
                    ]);
                } else {
                    // Payment failed
                    $transaction->update([
                        'status' => 'failed',
                        'mpesa_response' => json_encode($callbackData)
                    ]);

                    \Log::warning('Savings deposit failed', [
                        'transaction_id' => $transaction->id,
                        'result_code' => $resultCode,
                        'result_desc' => $resultDesc
                    ]);
                }
            }
        }

        // Return success response to M-Pesa
        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Callback received successfully']);
    }
}
