<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\MpesaCredential;

class MpesaCredentialController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $mpesaCredentials = MpesaCredential::all();

        return view('adminlte.mpesa-credentials.index', compact('mpesaCredentials'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('adminlte.mpesa-credentials.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'consumer_key' => 'required|string|max:255',
            'consumer_secret' => 'required|string|max:255',
            'shortcode' => 'required|string|max:255',
            'passkey' => 'required|string|max:255',
            'environment' => 'required|in:sandbox,production',
            'is_active' => 'nullable|boolean',
        ]);

        // If setting as active, deactivate all other credentials
        if ($request->is_active) {
            MpesaCredential::where('is_active', true)->update(['is_active' => false]);
        }

        MpesaCredential::create([
            'consumer_key' => $request->consumer_key,
            'consumer_secret' => $request->consumer_secret,
            'shortcode' => $request->shortcode,
            'passkey' => $request->passkey,
            'environment' => $request->environment,
            'is_active' => $request->is_active ?? false,
        ]);

        return redirect()->route('mpesa-credentials.index')
                        ->with('success', 'M-Pesa credentials added successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(MpesaCredential $mpesaCredential)
    {
        return view('adminlte.mpesa-credentials.show', compact('mpesaCredential'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(MpesaCredential $mpesaCredential)
    {
        return view('adminlte.mpesa-credentials.edit', compact('mpesaCredential'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, MpesaCredential $mpesaCredential)
    {
        $request->validate([
            'consumer_key' => 'required|string|max:255',
            'consumer_secret' => 'required|string|max:255',
            'shortcode' => 'required|string|max:255',
            'passkey' => 'required|string|max:255',
            'environment' => 'required|in:sandbox,production',
            'is_active' => 'nullable|boolean',
        ]);

        // If setting as active, deactivate all other credentials
        if ($request->is_active) {
            MpesaCredential::where('is_active', true)
                          ->where('id', '!=', $mpesaCredential->id)
                          ->update(['is_active' => false]);
        }

        $mpesaCredential->update([
            'consumer_key' => $request->consumer_key,
            'consumer_secret' => $request->consumer_secret,
            'shortcode' => $request->shortcode,
            'passkey' => $request->passkey,
            'environment' => $request->environment,
            'is_active' => $request->is_active ?? false,
        ]);

        return redirect()->route('mpesa-credentials.index')
                        ->with('success', 'M-Pesa credentials updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(MpesaCredential $mpesaCredential)
    {
        $mpesaCredential->delete();

        return redirect()->route('mpesa-credentials.index')
                        ->with('success', 'M-Pesa credentials deleted successfully!');
    }

    /**
     * Test M-Pesa connection
     */
    public function testConnection(MpesaCredential $mpesaCredential)
    {
        try {
            // Here you would implement actual M-Pesa connection test
            // For now, we'll just return a success response

            return response()->json([
                'success' => true,
                'message' => 'M-Pesa connection test successful! Credentials are valid.'
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Connection test failed: ' . $e->getMessage()
            ]);
        }
    }
}
