<?php

namespace App\Http\Controllers;

use App\Models\Occupation;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class OccupationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $occupations = Occupation::ordered()->paginate(15);

        return view('adminlte.occupations.index', compact('occupations'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('adminlte.occupations.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255|unique:occupations,name',
            'description' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0'
        ]);

        Occupation::create($request->all());

        return redirect()->route('occupations.index')
                        ->with('success', 'Occupation created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Occupation $occupation)
    {
        return view('adminlte.occupations.show', compact('occupation'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Occupation $occupation)
    {
        return view('adminlte.occupations.edit', compact('occupation'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Occupation $occupation)
    {
        $request->validate([
            'name' => ['required', 'string', 'max:255', Rule::unique('occupations')->ignore($occupation->id)],
            'description' => 'nullable|string|max:1000',
            'is_active' => 'boolean',
            'sort_order' => 'integer|min:0'
        ]);

        $occupation->update($request->all());

        return redirect()->route('occupations.index')
                        ->with('success', 'Occupation updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Occupation $occupation)
    {
        // Check if occupation is being used by any memberships
        $membershipsCount = \App\Models\Membership::where('occupation', $occupation->name)->count();
        if ($membershipsCount > 0) {
            return redirect()->route('occupations.index')
                            ->with('error', 'Cannot delete occupation that is being used by ' . $membershipsCount . ' member(s).');
        }

        $occupation->delete();

        return redirect()->route('occupations.index')
                        ->with('success', 'Occupation deleted successfully!');
    }

    /**
     * Toggle active status of occupation
     */
    public function toggleStatus(Occupation $occupation)
    {
        $occupation->update(['is_active' => !$occupation->is_active]);

        $status = $occupation->is_active ? 'activated' : 'deactivated';

        return redirect()->route('occupations.index')
                        ->with('success', "Occupation {$status} successfully!");
    }

    /**
     * Export occupations to PDF
     */
    public function exportPdf()
    {
        $occupations = Occupation::ordered()->get()->map(function ($occupation) {
            $occupation->memberships_count = \App\Models\Membership::where('occupation', $occupation->name)->count();
            return $occupation;
        });

        $pdf = \Barryvdh\DomPDF\Facade\Pdf::loadView('adminlte.occupations.exports.pdf', compact('occupations'))
                      ->setPaper('a4', 'landscape');

        return $pdf->download('occupations_report_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Export occupations to Excel
     */
    public function exportExcel()
    {
        return \Maatwebsite\Excel\Facades\Excel::download(new \App\Exports\OccupationsExport, 'occupations_report_' . date('Y-m-d_H-i-s') . '.xlsx');
    }

    /**
     * Export occupations to CSV
     */
    public function exportCsv()
    {
        return \Maatwebsite\Excel\Facades\Excel::download(new \App\Exports\OccupationsExport, 'occupations_report_' . date('Y-m-d_H-i-s') . '.csv');
    }
}
