<?php

namespace App\Http\Controllers;

use App\Models\Saving;
use App\Models\User;
use App\Models\SavingsTransaction;
use Illuminate\Http\Request;
use Barryvdh\DomPDF\Facade\Pdf;
use Maatwebsite\Excel\Facades\Excel;

class SavingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $savings = Saving::with(['user', 'transactions'])->paginate(15);

        // Calculate dynamic statistics
        $totalAccounts = Saving::count();
        $activeAccounts = Saving::where('status', 'active')->count();
        $maturedAccounts = Saving::where('status', 'matured')->count();
        $withdrawnAccounts = Saving::where('status', 'withdrawn')->count();

        // Total current savings balance (from active and matured accounts)
        $activeMaturedSavings = Saving::with('transactions')->whereIn('status', ['active', 'matured'])->get();
        $totalCurrentSavings = $activeMaturedSavings->sum(function ($saving) {
            return $saving->current_balance;
        });

        // Total withdrawn amount (sum of all withdrawal transactions)
        $totalWithdrawnAmount = \App\Models\SavingsTransaction::where('type', 'withdrawal')->sum('amount');

        // Total initial savings amount (sum of all account amounts)
        $totalInitialSavings = Saving::sum('amount');

        // Monthly savings growth (deposits - withdrawals for current month)
        $currentMonth = now()->format('Y-m');
        $monthlyDeposits = \App\Models\SavingsTransaction::where('type', 'deposit')
            ->whereRaw("DATE_FORMAT(transaction_date, '%Y-%m') = ?", [$currentMonth])
            ->sum('amount');

        $monthlyWithdrawals = \App\Models\SavingsTransaction::where('type', 'withdrawal')
            ->whereRaw("DATE_FORMAT(transaction_date, '%Y-%m') = ?", [$currentMonth])
            ->sum('amount');

        $monthlyNetGrowth = $monthlyDeposits - $monthlyWithdrawals;

        // Find member with most withdrawals (total withdrawal amount)
        $memberWithMostWithdrawals = \App\Models\SavingsTransaction::with(['account.user'])
            ->where('type', 'withdrawal')
            ->selectRaw('saving_id, SUM(amount) as total_withdrawn')
            ->groupBy('saving_id')
            ->orderBy('total_withdrawn', 'desc')
            ->first();

        $topWithdrawer = $memberWithMostWithdrawals ? $memberWithMostWithdrawals->account : null;
        $topWithdrawerAmount = $memberWithMostWithdrawals ? $memberWithMostWithdrawals->total_withdrawn : 0;

        // Recent transactions count (last 30 days)
        $recentTransactions = \App\Models\SavingsTransaction::where('transaction_date', '>=', now()->subDays(30))->count();

        // Find member with highest savings balance
        $topSaver = Saving::with(['user', 'transactions'])
            ->whereIn('status', ['active', 'matured'])
            ->get()
            ->map(function ($saving) {
                $saving->calculated_balance = $saving->transactions->sum(function ($transaction) {
                    return $transaction->type === 'deposit' ? $transaction->amount : -$transaction->amount;
                });
                return $saving;
            })
            ->sortByDesc('calculated_balance')
            ->first();

        // Find most recent saver (member with most recent transaction)
        $mostRecentTransaction = \App\Models\SavingsTransaction::with(['account.user'])
            ->orderBy('transaction_date', 'desc')
            ->first();

        $mostRecentSaver = $mostRecentTransaction ? $mostRecentTransaction->account : null;

        return view('adminlte.savings.index', compact(
            'savings',
            'totalAccounts',
            'activeAccounts',
            'maturedAccounts',
            'withdrawnAccounts',
            'totalCurrentSavings',
            'totalWithdrawnAmount',
            'totalInitialSavings',
            'monthlyDeposits',
            'monthlyWithdrawals',
            'monthlyNetGrowth',
            'recentTransactions',
            'topSaver',
            'mostRecentSaver',
            'topWithdrawer',
            'topWithdrawerAmount'
        ));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $users = User::whereHas('membership')->get();

        return view('adminlte.savings.create', compact('users'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
            'amount' => 'required|numeric|min:100',
            'type' => 'required|in:regular,fixed,emergency',
            'maturity_date' => 'nullable|date|after:today',
            'notes' => 'nullable|string|max:1000',
        ]);

        Saving::create($request->all());

        return redirect()->route('savings.index')
                        ->with('success', 'Savings account created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $saving = Saving::with(['user', 'transactions.processor'])->findOrFail($id);

        return view('adminlte.savings.show', compact('saving'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $saving = Saving::findOrFail($id);
        $users = User::whereHas('membership')->get();

        return view('adminlte.savings.edit', compact('saving', 'users'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $saving = Saving::findOrFail($id);

        $request->validate([
            'user_id' => 'required|exists:users,id',
            'type' => 'required|in:regular,fixed,emergency',
            'status' => 'required|in:active,withdrawn,matured',
            'maturity_date' => 'nullable|date',
            'notes' => 'nullable|string|max:1000',
        ]);

        // Only update fields that can be changed (exclude amount as it's read-only)
        $saving->update($request->only(['user_id', 'type', 'status', 'maturity_date', 'notes']));

        return redirect()->route('savings.index')
                        ->with('success', 'Savings account updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $saving = Saving::findOrFail($id);

        // Delete all related transactions first to avoid foreign key constraint violations
        $saving->transactions()->delete();

        // Now delete the savings account
        $saving->delete();

        return redirect()->route('savings.index')
                        ->with('success', 'Savings account and all associated transactions deleted successfully!');
    }

    /**
     * Withdraw savings.
     */
    public function withdraw(Saving $saving)
    {
        $saving->update(['status' => 'withdrawn']);

        return redirect()->route('savings.index')
                        ->with('success', 'Savings withdrawn successfully!');
    }

    /**
     * Mark savings as matured.
     */
    public function mature(Saving $saving)
    {
        $saving->update(['status' => 'matured']);

        return redirect()->route('savings.index')
                        ->with('success', 'Savings marked as matured successfully!');
    }

    /**
     * Show deposit form.
     */
    public function showDepositForm(Saving $saving)
    {
        return view('adminlte.savings.deposit', compact('saving'));
    }

    /**
     * Process deposit.
     */
    public function deposit(Request $request, Saving $saving)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'notes' => 'nullable|string|max:1000',
        ]);

        $currentBalance = $saving->current_balance;
        $newBalance = $currentBalance + $request->amount;

        // Create transaction record
        $transaction = SavingsTransaction::create([
            'saving_id' => $saving->id,
            'type' => 'deposit',
            'amount' => $request->amount,
            'balance_before' => $currentBalance,
            'balance_after' => $newBalance,
            'transaction_number' => 'DEP-' . date('YmdHis') . '-' . $saving->id,
            'notes' => $request->notes,
            'processed_by' => auth()->id(),
            'transaction_date' => now(),
        ]);

        return redirect()->route('savings.transaction.receipt', $transaction)
                        ->with('success', 'Deposit processed successfully!');
    }

    /**
     * Show withdrawal form.
     */
    public function showWithdrawalForm(Saving $saving)
    {
        return view('adminlte.savings.withdraw', compact('saving'));
    }

    /**
     * Process withdrawal.
     */
    public function processWithdrawal(Request $request, Saving $saving)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1|max:' . $saving->current_balance,
            'notes' => 'nullable|string|max:1000',
        ]);

        $currentBalance = $saving->current_balance;
        $newBalance = $currentBalance - $request->amount;

        // Create transaction record
        $transaction = SavingsTransaction::create([
            'saving_id' => $saving->id,
            'type' => 'withdrawal',
            'amount' => $request->amount,
            'balance_before' => $currentBalance,
            'balance_after' => $newBalance,
            'transaction_number' => 'WIT-' . date('YmdHis') . '-' . $saving->id,
            'notes' => $request->notes,
            'processed_by' => auth()->id(),
            'transaction_date' => now(),
        ]);

        return redirect()->route('savings.transaction.receipt', $transaction)
                        ->with('success', 'Withdrawal processed successfully!');
    }

    /**
     * Show transaction receipt.
     */
    public function showReceipt(SavingsTransaction $transaction)
    {
        $transaction->load('account.user', 'processor');
        return view('adminlte.savings.receipt', compact('transaction'));
    }

    /**
     * Print transaction receipt.
     */
    public function printReceipt(SavingsTransaction $transaction)
    {
        $transaction->load('account.user', 'processor');

        $pdf = Pdf::loadView('adminlte.savings.exports.receipt_pdf', compact('transaction'))
                  ->setPaper('a6', 'portrait');

        return $pdf->download('savings_receipt_' . $transaction->transaction_number . '.pdf');
    }

    /**
     * Export savings to PDF
     */
    public function exportPdf()
    {
        $savings = Saving::with('user')->get();

        $pdf = Pdf::loadView('adminlte.savings.exports.pdf', compact('savings'))
                  ->setPaper('a4', 'landscape');

        return $pdf->download('savings_report_' . date('Y-m-d_H-i-s') . '.pdf');
    }

    /**
     * Export savings to Excel
     */
    public function exportExcel()
    {
        return Excel::download(new \App\Exports\SavingsExport, 'savings_report_' . date('Y-m-d_H-i-s') . '.xlsx');
    }

    /**
     * Export savings to CSV
     */
    public function exportCsv()
    {
        return Excel::download(new \App\Exports\SavingsExport, 'savings_report_' . date('Y-m-d_H-i-s') . '.csv');
    }

    /**
     * Download savings plan for a specific account
     */
    public function downloadSavingsPlan(Saving $saving)
    {
        $saving->load(['user', 'transactions.processor']);

        $pdf = Pdf::loadView('adminlte.savings.exports.savings-plan', compact('saving'))
                  ->setPaper('a4', 'portrait');

        return $pdf->download('savings_plan_' . $saving->user->full_name . '_' . $saving->id . '.pdf');
    }
}
