<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\HasOne;

class LoanGuarantee extends Model
{
    protected $fillable = [
        'loan_id',
        'guarantee_type',
        'status',
        'notes',
        'verification_data',
        'verified_at',
        'verified_by',
    ];

    protected $casts = [
        'verification_data' => 'array',
        'verified_at' => 'datetime',
    ];

    public function loan(): BelongsTo
    {
        return $this->belongsTo(Loan::class);
    }

    public function verifiedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'verified_by');
    }

    public function guarantors(): HasMany
    {
        return $this->hasMany(LoanGuarantor::class);
    }

    public function collateral(): HasMany
    {
        return $this->hasMany(LoanCollateral::class);
    }

    public function salaryGuarantee(): HasOne
    {
        return $this->hasOne(LoanSalaryGuarantee::class);
    }

    public function getTypeLabelAttribute()
    {
        return match($this->guarantee_type) {
            'salary' => 'Salary Guarantee',
            'collateral' => 'Collateral Guarantee',
            'guarantor' => 'Guarantor Guarantee',
            default => ucfirst($this->guarantee_type)
        };
    }

    public function getStatusColorAttribute()
    {
        return match($this->status) {
            'approved' => 'success',
            'pending' => 'warning',
            'rejected' => 'danger',
            default => 'secondary'
        };
    }
}
