@extends('adminlte::page')

@section('title', 'Create Money Market Account')

@section('content')
<div class="container-fluid">
    <div class="row">
        <div class="col-md-8 offset-md-2">
            <div class="card">
                <div class="card-header">
                    <h3 class="card-title">Create Money Market Account</h3>
                    <div class="card-tools">
                        <a href="{{ route('money-market-accounts.index') }}" class="btn btn-secondary btn-sm">
                            <i class="fas fa-arrow-left"></i> Back
                        </a>
                    </div>
                </div>

                <form action="{{ route('money-market-accounts.store') }}" method="POST">
                    @csrf
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="membership_id">Member <span class="text-danger">*</span></label>
                                    <select name="membership_id" id="membership_id" class="form-control @error('membership_id') is-invalid @enderror" required>
                                        <option value="">Select Member</option>
                                        @foreach($memberships as $membership)
                                        <option value="{{ $membership->id }}" {{ old('membership_id') == $membership->id ? 'selected' : '' }}>
                                            {{ $membership->user->full_name ?? 'N/A' }} ({{ $membership->membership_number }})
                                        </option>
                                        @endforeach
                                    </select>
                                    @error('membership_id')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="principal_amount">Principal Amount <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <div class="input-group-prepend">
                                            <span class="input-group-text">{{ \App\Helpers\CurrencyHelper::getSymbol('KES') }}</span>
                                        </div>
                                        <input type="number" name="principal_amount" id="principal_amount" class="form-control @error('principal_amount') is-invalid @enderror"
                                               value="{{ old('principal_amount') }}" min="1000" step="0.01" required>
                                    </div>
                                    @error('principal_amount')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                    <small class="form-text text-muted">Minimum amount: {{ \App\Helpers\CurrencyHelper::format(1000, 'KES') }}</small>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="interest_rate">Interest Rate (%) <span class="text-danger">*</span></label>
                                    <div class="input-group">
                                        <input type="number" name="interest_rate" id="interest_rate" class="form-control @error('interest_rate') is-invalid @enderror"
                                               value="{{ old('interest_rate', 8.50) }}" min="0" max="50" step="0.01" required>
                                        <div class="input-group-append">
                                            <span class="input-group-text">%</span>
                                        </div>
                                    </div>
                                    @error('interest_rate')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="interest_type">Interest Type <span class="text-danger">*</span></label>
                                    <select name="interest_type" id="interest_type" class="form-control @error('interest_type') is-invalid @enderror" required>
                                        <option value="simple" {{ old('interest_type', 'simple') == 'simple' ? 'selected' : '' }}>Simple Interest</option>
                                        <option value="compound" {{ old('interest_type') == 'compound' ? 'selected' : '' }}>Compound Interest</option>
                                    </select>
                                    @error('interest_type')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="compounding_frequency">Compounding Frequency</label>
                                    <select name="compounding_frequency" id="compounding_frequency" class="form-control @error('compounding_frequency') is-invalid @enderror">
                                        <option value="annually" {{ old('compounding_frequency', 'annually') == 'annually' ? 'selected' : '' }}>Annually</option>
                                        <option value="quarterly" {{ old('compounding_frequency') == 'quarterly' ? 'selected' : '' }}>Quarterly</option>
                                        <option value="monthly" {{ old('compounding_frequency') == 'monthly' ? 'selected' : '' }}>Monthly</option>
                                    </select>
                                    <small class="form-text text-muted">Only applies to compound interest</small>
                                    @error('compounding_frequency')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="term_months">Term (Months) <span class="text-danger">*</span></label>
                                    <select name="term_months" id="term_months" class="form-control @error('term_months') is-invalid @enderror" required>
                                        <option value="">Select Term</option>
                                        <option value="3" {{ old('term_months') == '3' ? 'selected' : '' }}>3 Months</option>
                                        <option value="6" {{ old('term_months') == '6' ? 'selected' : '' }}>6 Months</option>
                                        <option value="12" {{ old('term_months') == '12' ? 'selected' : '' }}>12 Months</option>
                                        <option value="24" {{ old('term_months') == '24' ? 'selected' : '' }}>24 Months</option>
                                        <option value="36" {{ old('term_months') == '36' ? 'selected' : '' }}>36 Months</option>
                                        <option value="48" {{ old('term_months') == '48' ? 'selected' : '' }}>48 Months</option>
                                        <option value="60" {{ old('term_months') == '60' ? 'selected' : '' }}>60 Months</option>
                                    </select>
                                    @error('term_months')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="start_date">Start Date <span class="text-danger">*</span></label>
                                    <input type="date" name="start_date" id="start_date" class="form-control @error('start_date') is-invalid @enderror"
                                           value="{{ old('start_date', date('Y-m-d')) }}" min="{{ date('Y-m-d') }}" required>
                                    @error('start_date')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="currency_code">Currency <span class="text-danger">*</span></label>
                                    <select name="currency_code" id="currency_code" class="form-control @error('currency_code') is-invalid @enderror" required>
                                        <option value="KES" {{ old('currency_code', 'KES') == 'KES' ? 'selected' : '' }}>Kenyan Shilling (KES)</option>
                                        <option value="USD" {{ old('currency_code') == 'USD' ? 'selected' : '' }}>US Dollar (USD)</option>
                                        <option value="EUR" {{ old('currency_code') == 'EUR' ? 'selected' : '' }}>Euro (EUR)</option>
                                    </select>
                                    @error('currency_code')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-12">
                                <div class="form-group">
                                    <label for="notes">Notes</label>
                                    <textarea name="notes" id="notes" class="form-control @error('notes') is-invalid @enderror" rows="3"
                                              placeholder="Optional notes about this investment">{{ old('notes') }}</textarea>
                                    @error('notes')
                                        <div class="invalid-feedback">{{ $message }}</div>
                                    @enderror
                                </div>
                            </div>
                        </div>

                        <!-- Preview Section -->
                        <div class="row mt-4">
                            <div class="col-md-12">
                                <div class="card border-info">
                                    <div class="card-header bg-info">
                                        <h5 class="card-title mb-0">Investment Preview</h5>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-6">
                                                <strong>Principal Amount:</strong> <span id="preview-principal">-</span><br>
                                                <strong>Interest Rate:</strong> <span id="preview-rate">-</span><br>
                                                <strong>Term:</strong> <span id="preview-term">-</span>
                                            </div>
                                            <div class="col-md-6">
                                                <strong>Start Date:</strong> <span id="preview-start">-</span><br>
                                                <strong>Maturity Date:</strong> <span id="preview-maturity">-</span><br>
                                                <strong>Estimated Interest:</strong> <span id="preview-interest">-</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <div class="card-footer">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Create Account
                        </button>
                        <a href="{{ route('money-market-accounts.index') }}" class="btn btn-secondary ml-2">
                            <i class="fas fa-times"></i> Cancel
                        </a>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@section('js')
<script>
$(document).ready(function() {
    // Toggle compounding frequency field based on interest type
    function toggleCompoundingFrequency() {
        const interestType = $('#interest_type').val();
        const compoundingField = $('#compounding_frequency').closest('.form-group');

        if (interestType === 'compound') {
            compoundingField.show();
        } else {
            compoundingField.hide();
            $('#compounding_frequency').val('annually'); // Reset to default
        }
    }

    // Update preview when form values change
    function updatePreview() {
        const principal = parseFloat($('#principal_amount').val()) || 0;
        const rate = parseFloat($('#interest_rate').val()) || 0;
        const term = parseInt($('#term_months').val()) || 0;
        const startDate = $('#start_date').val();
        const currency = $('#currency_code').val();
        const interestType = $('#interest_type').val();
        const compoundingFrequency = $('#compounding_frequency').val();

        if (principal > 0 && rate > 0 && term > 0 && startDate) {
            const maturityDate = new Date(startDate);
            maturityDate.setMonth(maturityDate.getMonth() + term);

            let interest = 0;
            const years = term / 12;

            if (interestType === 'compound') {
                // Compound interest calculation
                const compoundsPerYear = compoundingFrequency === 'monthly' ? 12 :
                                       compoundingFrequency === 'quarterly' ? 4 : 1;
                interest = principal * (Math.pow(1 + (rate / 100) / compoundsPerYear, compoundsPerYear * years) - 1);
            } else {
                // Simple interest calculation
                interest = principal * (rate / 100) * years;
            }

            $('#preview-principal').text(formatCurrency(principal, currency));
            $('#preview-rate').text(rate + '% per annum (' + (interestType === 'compound' ? 'Compound' : 'Simple') + ')');
            $('#preview-term').text(term + ' months');
            $('#preview-start').text(new Date(startDate).toLocaleDateString());
            $('#preview-maturity').text(maturityDate.toLocaleDateString());
            $('#preview-interest').text(formatCurrency(interest, currency));
        } else {
            $('#preview-principal, #preview-rate, #preview-term, #preview-start, #preview-maturity, #preview-interest').text('-');
        }
    }

    function formatCurrency(amount, currency) {
        const symbols = { 'KES': 'KSh', 'USD': '$', 'EUR': '€' };
        return symbols[currency] + ' ' + amount.toLocaleString();
    }

    // Bind events
    $('#principal_amount, #interest_rate, #term_months, #start_date, #currency_code, #interest_type, #compounding_frequency').on('input change', function() {
        updatePreview();
        if ($(this).attr('id') === 'interest_type') {
            toggleCompoundingFrequency();
        }
    });

    // Initialize
    toggleCompoundingFrequency();
    updatePreview();
});
</script>
@endsection
